"""
FFT 分析模块（用于后续实时显示频谱/波形）。

设计目标：
    - 支持多通道（1..4 或更多）流式输入
    - 内部维护每通道缓冲区，凑够 N 点后做一次 FFT
    - 输出适合画图的数据：频率轴 freqs + 幅度谱 magnitude（可选相位）

说明：
    - 这里不负责画图，只负责计算 FFT
    - 输入建议为 int16 采样值序列（list/np.ndarray），也支持 float
    - 目前使用 numpy 进行 FFT；如果运行环境没有 numpy，会抛出明确错误
"""

from __future__ import annotations

from dataclasses import dataclass
import threading
from typing import Dict, Iterable, Optional, Union

import numpy as np


NumberArrayLike = Union[Iterable[int], Iterable[float]]


@dataclass(frozen=True)
class FFTResult:
    """一次 FFT 的输出结果（单通道）。"""

    freqs: "np.ndarray"  # shape: (N/2+1,)
    magnitude: "np.ndarray"  # shape: (N/2+1,)
    phase: Optional["np.ndarray"]  # shape: (N/2+1,) or None


class FFTAnalyzer:
    """
    多通道流式 FFT 分析器。

    你后续做实时显示时，只需要周期性调用：
        - append_samples(channel, samples)
        - compute_if_ready(channel)  或 compute_latest(channel)
    """

    def __init__(
        self,
        sample_rate: float,
        fft_size: int = 2048,
        *,
        return_phase: bool = False,
        magnitude_scale: str = "linear",
    ) -> None:
        """
        Args:
            sample_rate: 采样率（Hz）
            fft_size: FFT 点数 N（建议 2 的幂）
            重叠率: 固定为 50%（hop_size = fft_size/2）
            窗函数: 固定为汉宁窗（Hann），窗长 = fft_size
            return_phase: 是否返回相位谱
            magnitude_scale: 'linear' 或 'db'
        """
        if sample_rate <= 0:
            raise ValueError("sample_rate 必须 > 0")
        if fft_size <= 0:
            raise ValueError("fft_size 必须为正整数")
        if fft_size % 2 != 0:
            raise ValueError("fft_size 必须为偶数（50% 重叠需要 hop_size=fft_size/2 为整数）")

        self.sample_rate = float(sample_rate)
        self.fft_size = int(fft_size)
        # 50% 重叠：每次 FFT 后前进 N/2 个点
        self.hop_size = self.fft_size // 2

        # 固定汉宁窗（窗长 = fft_size）
        self.window = np.hanning(self.fft_size).astype(np.float32)
        self.return_phase = bool(return_phase)
        self.magnitude_scale = magnitude_scale

        # 每个通道一个缓冲区（float32）
        self._buffers: Dict[int, "np.ndarray"] = {}

        # 每个通道最新一次 FFT 结果（用于异步读取：绘图/报警等）
        self._last_results: Dict[int, FFTResult] = {}
        self._result_lock = threading.Lock()

        # 预计算频率轴
        self._freqs = np.fft.rfftfreq(self.fft_size, d=1.0 / self.sample_rate)

    def append_samples(self, channel: int, samples: NumberArrayLike) -> None:
        """向指定通道追加采样数据。"""
        x = self._to_float_array(samples)
        if x.size == 0:
            return
        buf = self._buffers.get(channel)
        self._buffers[channel] = x if buf is None else np.concatenate([buf, x], axis=0)

    def compute_if_ready(self, channel: int) -> Optional[FFTResult]:
        """
        如果缓冲区 >= fft_size，则计算一次 FFT 并返回结果；否则返回 None。

        计算后会从缓冲区丢弃 hop_size 个采样点（实现滑窗/重叠分析）。
        """
        buf = self._buffers.get(channel)
        if buf is None or buf.size < self.fft_size:
            return None

        frame = buf[: self.fft_size]
        self._buffers[channel] = buf[self.hop_size :]
        result = self._compute_frame(frame)
        with self._result_lock:
            self._last_results[channel] = result
        return result

    def compute_latest(self, channel: int) -> Optional[FFTResult]:
        """用最新的 fft_size 点做 FFT（不移动缓冲区），适合“刷新式”绘图。"""
        buf = self._buffers.get(channel)
        if buf is None or buf.size < self.fft_size:
            return None
        frame = buf[-self.fft_size :]
        result = self._compute_frame(frame)
        with self._result_lock:
            self._last_results[channel] = result
        return result

    def get_freqs(self) -> "np.ndarray":
        """频率轴（Hz），shape=(N/2+1,)"""
        return self._freqs

    def get_last_result(self, channel: int) -> Optional[FFTResult]:
        """
        获取该通道“最近一次计算得到的 FFT 结果”。

        适用场景：
            - 报警模块按较低频率（例如 5-20Hz）异步读取最新频谱
            - 绘图模块与报警模块解耦
        """
        with self._result_lock:
            return self._last_results.get(channel)

    # -------- internal --------
    def _compute_frame(self, frame: "np.ndarray") -> FFTResult:
        xw = frame * self.window
        spec = np.fft.rfft(xw)

        # 幅值归一化/校正（让阈值更“可用”）：
        # - 当前固定 Hann 窗，使用 sum(window) 做相干增益校正
        # - 输出单边幅度谱：除 DC 与 Nyquist 外，其它频点乘 2
        coherent_gain = float(np.sum(self.window))
        if coherent_gain <= 0:
            raise ValueError("window sum 非法，无法做幅值归一化")

        mag = np.abs(spec) / coherent_gain
        if mag.size >= 3:
            mag[1:-1] *= 2.0
        if self.magnitude_scale == "db":
            eps = np.finfo(np.float32).eps
            mag = 20.0 * np.log10(mag + eps)
        elif self.magnitude_scale != "linear":
            raise ValueError("magnitude_scale 只能是 'linear' 或 'db'")

        phase = np.angle(spec) if self.return_phase else None
        return FFTResult(freqs=self._freqs, magnitude=mag, phase=phase)

    def _to_float_array(self, samples: NumberArrayLike) -> "np.ndarray":
        # 这里不做单位/缩放换算；只是把输入转成 float32
        return np.asarray(list(samples), dtype=np.float32)

