"""
UDP 数据接收模块

协议约定（当前约束）：
    - 远端下位机 IP: 192.168.19.140
    - 远端端口:      8089
    - 每个 UDP 包大小: 1044 字节（本模块只保证完整接收并缓存该数据包）
"""

import logging
import socket
import threading
import time
from collections import deque
from typing import Deque, Dict, Optional, Tuple, Any


logger = logging.getLogger(__name__)


class UDPReceiver:
    """
    下位机 UDP 数据接收器

    功能（仅负责“接收”与“缓存”，不做解析）：
        - 绑定主机的 8089 端口接收 UDP 数据
        - 只接收来自指定下位机 IP 的数据（192.168.19.140）
        - 校验包长为 1044 字节
        - 将完整数据包推入缓冲队列，供后续业务模块做解析 / FFT 等处理
    """

    def __init__(
        self,
        listen_host: str = "0.0.0.0",
        listen_port: int = 8089,
        device_ip: str = "192.168.19.140",
        packet_size: int = 1044,
        buffer_size: int = 2000,
    ) -> None:
        """
        Args:
            listen_host: 主机监听地址，一般保持 0.0.0.0
            listen_port: 主机监听端口，按需求为 8089
            device_ip:   期望数据来源的下位机 IP
            packet_size: UDP 包总长度，字节（当前为 1044）
            buffer_size: 内部缓冲区最多缓存多少个数据包
        """
        self.listen_host = listen_host
        self.listen_port = listen_port
        self.device_ip = device_ip
        self.packet_size = packet_size

        self._sock: Optional[socket.socket] = None
        self._thread: Optional[threading.Thread] = None
        self._running = False

        self._buffer: Deque[Dict[str, Any]] = deque(maxlen=buffer_size)
        self._buffer_lock = threading.Lock()

        # 统计信息
        self._total_packets = 0
        self._invalid_packets = 0
        self._start_time: Optional[float] = None

        logger.info(
            f"UDPReceiver 初始化: 监听 {listen_host}:{listen_port}, "
            f"下位机 IP={device_ip}, 包长={packet_size}"
        )

    # ================= 公共接口 =================
    def start(self) -> bool:
        """启动接收线程并开始监听 UDP 数据。"""
        if self._running:
            return True

        try:
            self._sock = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
            # 端口复用，方便多次启动/停止
            self._sock.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
            # 设置 1 秒超时，便于线程优雅退出
            self._sock.settimeout(1.0)
            # 绑定到主机 8089 端口
            self._sock.bind((self.listen_host, self.listen_port))

            self._running = True
            self._start_time = time.time()

            self._thread = threading.Thread(
                target=self._receive_loop,
                name="UDPReceiverThread",
                daemon=True,
            )
            self._thread.start()

            logger.info(
                f"UDPReceiver 已启动，正在监听 {self.listen_host}:{self.listen_port}"
            )
            return True
        except Exception as exc:  # noqa: BLE001
            logger.error(f"启动 UDPReceiver 失败: {exc}")
            self.stop()
            return False

    def stop(self) -> None:
        """停止接收并关闭 socket。"""
        self._running = False

        if self._thread and self._thread.is_alive():
            self._thread.join(timeout=2.0)

        if self._sock:
            try:
                self._sock.close()
            finally:
                self._sock = None

        logger.info("UDPReceiver 已停止")

    def get_packet(self, timeout: float = 0.1) -> Optional[Dict[str, Any]]:
        """
        从缓冲区取出一个最新的数据包。

        Args:
            timeout: 等待时间（秒），避免阻塞太久

        Returns:
            dict | None:
                {
                    "timestamp": float,        # 接收时间戳
                    "raw_data": bytes,         # 完整 1044 字节原始数据
                    "source_addr": (ip, port), # 远端地址
                    "index": int,              # 包序号
                }
        """
        end_time = time.time() + timeout
        while time.time() < end_time:
            with self._buffer_lock:
                if self._buffer:
                    return self._buffer.popleft()
            time.sleep(0.001)
        return None

    def get_stats(self) -> Dict[str, Any]:
        """返回当前接收统计信息。"""
        now = time.time()
        elapsed = (now - self._start_time) if self._start_time else 0.0
        with self._buffer_lock:
            buffer_len = len(self._buffer)

        return {
            "total_packets": self._total_packets,
            "invalid_packets": self._invalid_packets,
            "buffer_size": buffer_len,
            "elapsed": elapsed,
            "packets_per_second": (
                self._total_packets / elapsed if elapsed > 0 else 0.0
            ),
        }

    # ================= 内部实现 =================
    def _receive_loop(self) -> None:
        assert self._sock is not None
        logger.info("UDPReceiver 接收线程已启动，等待数据...")

        while self._running:
            try:
                data, addr = self._sock.recvfrom(self.packet_size * 2)
            except socket.timeout:
                continue
            except OSError:
                # 通常是 socket 已关闭
                break
            except Exception as exc:  # noqa: BLE001
                logger.error(f"接收数据异常: {exc}")
                continue

            src_ip, src_port = addr  # type: ignore[misc]

            # 过滤非指定下位机 IP 的数据
            if src_ip != self.device_ip:
                logger.debug(f"丢弃来自未知 IP {addr} 的数据包")
                continue

            # 校验包长
            if len(data) != self.packet_size:
                self._invalid_packets += 1
                logger.warning(
                    f"收到异常长度数据包: {len(data)} 字节，期望 {self.packet_size}"
                )
                continue

            self._total_packets += 1

            packet: Dict[str, Any] = {
                "timestamp": time.time(),
                "raw_data": data,
                "source_addr": addr,
                "index": self._total_packets,
            }

            with self._buffer_lock:
                self._buffer.append(packet)

        logger.info("UDPReceiver 接收线程退出")
        