"""
数据包解析模块

功能：
    - 从 UDPReceiver 接收原始数据包
    - 解析识别头，判断通道组（1/2 或 3/4）
    - 分离各通道的音频数据（每 2 字节为一个采样点，交替传输）

数据包格式（1044 字节）：
    - 字节 0-3: 识别头
        - 01 02 03 04: 通道 1、2 的数据
        - 05 06 07 08: 通道 3、4 的数据
    - 字节 4-19: 中间保留区/元数据（16 字节）
        - 第 5  字节：保留位（一般为 0x00，不处理）
        - 第 6-9 字节：通道 1/2/3/4 音量（各 1 字节，不区分识别头，每包都有）
        - 第 10-17 字节：通道 1/2/3/4 温度（各 2 字节，小端序 int16，不区分识别头，每包都有）
        - 第 18-20 字节：音频流缓存编号（3 字节；编号出现顺序为 1、3、2、4）
    - 字节 20-1043: 音频数据（1024 字节，包的最后 1024 字节）
        - 每 2 字节为一个采样点
        - 交替传输：通道A、通道B、通道A、通道B...
        - 共 512 个采样点，每个通道 256 个采样点
"""

import struct
from typing import Dict, Optional, Tuple

# 识别头常量
HEADER_CH12 = bytes([0x01, 0x02, 0x03, 0x04])  # 通道 1、2
HEADER_CH34 = bytes([0x05, 0x06, 0x07, 0x08])  # 通道 3、4


class ParsedPacket:
    """解析后的数据包结构"""

    def __init__(
        self,
        raw_data: bytes,
        header: bytes,
        channel_group: str,
        channel_a_data: bytes,
        channel_b_data: bytes,
        metadata: bytes,
        reserved_byte: int,
        volumes_raw: bytes,
        temperatures_raw: bytes,
        audio_cache_id_raw: bytes,
        timestamp: float,
        packet_index: int,
    ):
        """
        Args:
            raw_data: 原始 1044 字节数据
            header: 识别头（4 字节）
            channel_group: 通道组标识，'ch12' 或 'ch34'
            channel_a_data: 通道 A 的音频数据（256 个采样点，512 字节）
            channel_b_data: 通道 B 的音频数据（256 个采样点，512 字节）
            metadata: 字节 4-19 的中间保留区/元数据（16 字节）
            reserved_byte: 第 5 字节保留位（0-255），不做换算
            volumes_raw: 第 6-9 字节音量原始数据（4 字节，不做换算）
            temperatures_raw: 第 10-17 字节温度原始数据（8 字节，不做换算）
            audio_cache_id_raw: 字节 18-20 缓存编号原始值（3 字节）
            timestamp: 接收时间戳
            packet_index: 包序号
        """
        self.raw_data = raw_data
        self.header = header
        self.channel_group = channel_group
        self.channel_a_data = channel_a_data
        self.channel_b_data = channel_b_data
        self.metadata = metadata
        self.reserved_byte = reserved_byte
        self.volumes_raw = volumes_raw
        self.temperatures_raw = temperatures_raw
        self.audio_cache_id_raw = audio_cache_id_raw
        self.timestamp = timestamp
        self.packet_index = packet_index

    def get_channel_data(self, channel: int) -> Optional[bytes]:
        """
        根据通道号获取对应的音频数据。

        Args:
            channel: 通道号（1, 2, 3, 4）

        Returns:
            该通道的音频数据（bytes），如果通道号无效则返回 None
        """
        if self.channel_group == "ch12":
            if channel == 1:
                return self.channel_a_data
            elif channel == 2:
                return self.channel_b_data
        elif self.channel_group == "ch34":
            if channel == 3:
                return self.channel_a_data
            elif channel == 4:
                return self.channel_b_data
        return None

    def get_channel_samples(self, channel: int) -> Optional[list]:
        """
        将通道音频数据转换为采样点列表（每个采样点为 2 字节，小端序 int16）。

        Args:
            channel: 通道号（1, 2, 3, 4）

        Returns:
            采样点列表（int16 值），如果通道号无效则返回 None
        """
        data = self.get_channel_data(channel)
        if data is None:
            return None

        samples = []
        for i in range(0, len(data), 2):
            sample = struct.unpack("<h", data[i : i + 2])[0]  # 小端序 int16
            samples.append(sample)
        return samples


def parse_packet(packet_dict: Dict) -> Optional[ParsedPacket]:
    """
    解析从 UDPReceiver.get_packet() 获取的数据包字典。

    Args:
        packet_dict: UDPReceiver 返回的数据包字典，包含：
            - "raw_data": bytes，完整 1044 字节
            - "timestamp": float，接收时间戳
            - "index": int，包序号

    Returns:
        ParsedPacket 对象，如果数据包格式无效则返回 None
    """
    raw_data = packet_dict.get("raw_data")
    if raw_data is None or len(raw_data) != 1044:
        return None

    # 提取识别头（前 4 字节）
    header = raw_data[0:4]

    # 判断通道组
    if header == HEADER_CH12:
        channel_group = "ch12"
        channel_a_name = "通道1"
        channel_b_name = "通道2"
    elif header == HEADER_CH34:
        channel_group = "ch34"
        channel_a_name = "通道3"
        channel_b_name = "通道4"
    else:
        # 未知的识别头，返回 None
        return None

    # 提取中间保留区/元数据（字节 4-19，16 字节）
    metadata = raw_data[4:20]
    if len(metadata) != 16:
        return None

    # 解析元数据区（下面的“第 N 字节”均按 1 开始计数）：
    # - 第 5  字节：raw_data[4]      保留位
    # - 第 6-9 字节：raw_data[5:9]   4 路音量（4 bytes，原始值，不换算）
    # - 第 10-17 字节：raw_data[9:17] 4 路温度（8 bytes，原始值，不换算）
    # - 第 18-20 字节：raw_data[17:20] 缓存编号（3 bytes）
    reserved_byte = metadata[0]
    volumes_raw = metadata[1:5]
    temperatures_raw = metadata[5:13]
    audio_cache_id_raw = metadata[13:16]
    if len(volumes_raw) != 4 or len(temperatures_raw) != 8 or len(audio_cache_id_raw) != 3:
        return None

    # 提取音频数据（包的最后 1024 字节）
    audio_data = raw_data[20:]
    if len(audio_data) != 1024:
        return None

    # 分离两个通道的数据（交替传输）
    # 音频数据中，每 4 字节为一组：通道A(2字节) + 通道B(2字节)
    # 通道A: 索引 0, 4, 8, 12, ... (偶数索引的 2 字节)
    # 通道B: 索引 2, 6, 10, 14, ... (奇数索引的 2 字节)
    channel_a_bytes = bytearray()
    channel_b_bytes = bytearray()

    for i in range(0, len(audio_data), 4):
        # 每组 4 字节：前 2 字节是通道A，后 2 字节是通道B
        if i + 1 < len(audio_data):
            channel_a_bytes.extend(audio_data[i : i + 2])
        if i + 3 < len(audio_data):
            channel_b_bytes.extend(audio_data[i + 2 : i + 4])

    # 确保每个通道正好是 512 字节（256 个采样点）
    if len(channel_a_bytes) != 512 or len(channel_b_bytes) != 512:
        return None

    return ParsedPacket(
        raw_data=raw_data,
        header=header,
        channel_group=channel_group,
        channel_a_data=bytes(channel_a_bytes),
        channel_b_data=bytes(channel_b_bytes),
        metadata=metadata,
        reserved_byte=reserved_byte,
        volumes_raw=volumes_raw,
        temperatures_raw=temperatures_raw,
        audio_cache_id_raw=audio_cache_id_raw,
        timestamp=packet_dict.get("timestamp", 0.0),
        packet_index=packet_dict.get("index", 0),
    )


def separate_channels(audio_data: bytes) -> Tuple[bytes, bytes]:
    """
    从交替传输的音频数据中分离出两个通道的数据。

    Args:
        audio_data: 音频数据（1024 字节），交替传输格式

    Returns:
        (channel_a_data, channel_b_data): 两个通道的音频数据，各 512 字节
    """
    channel_a_bytes = bytearray()
    channel_b_bytes = bytearray()

    for i in range(0, len(audio_data), 4):
        if i + 1 < len(audio_data):
            channel_a_bytes.extend(audio_data[i : i + 2])
        if i + 3 < len(audio_data):
            channel_b_bytes.extend(audio_data[i + 2 : i + 4])

    return bytes(channel_a_bytes), bytes(channel_b_bytes)

