"""
报警演示脚本：对四路通道进行频带报警监测。

报警逻辑（最基础版本）：
    - 对每个通道，当 FFT 结果产生时，计算 30-80kHz 频段内幅值峰值 band_peak
    - 若 band_peak > threshold，则在终端打印该通道报警信息
"""

from __future__ import annotations

import time
from typing import Dict
import numpy as np


from data_acquisition.packet_parser import parse_packet
from data_acquisition.udp_receiver import UDPReceiver
from signal_processing.fft_analyzer import FFTAnalyzer


# 报警参数（写死，后续如需配置文件，在 main.py 里读取后再改造此模块）
_BAND_START_HZ = 30000.0
_BAND_END_HZ = 80000.0
_THRESHOLD = 150.0
_SAMPLE_RATE = 176200.0
_FFT_SIZE = 2048
_MIN_ALARM_INTERVAL_S = 0.2
_PACKET_TIMEOUT_S = 0.5


def _bytes_to_int16_samples(b: bytes) -> np.ndarray:
    # 不做换算，直接按小端 int16 解释
    return np.frombuffer(b, dtype="<i2")


def run_alarm_monitoring(
) -> None:
    """
    启动四路通道的频带报警监测（阻塞运行，Ctrl+C 退出）。
    """
    rx = UDPReceiver()
    if not rx.start():
        raise RuntimeError("UDPReceiver 启动失败")

    analyzer = FFTAnalyzer(
        sample_rate=_SAMPLE_RATE,
        fft_size=_FFT_SIZE,
        magnitude_scale="linear",
    )
    freqs = analyzer.get_freqs()
    band_mask = (freqs >= _BAND_START_HZ) & (freqs <= _BAND_END_HZ)

    print(
        "开始报警监测（Ctrl+C 退出）...\n"
        f"- band: {_BAND_START_HZ/1000:.0f}kHz ~ {_BAND_END_HZ/1000:.0f}kHz\n"
        f"- threshold: {_THRESHOLD}\n"
        f"- sample_rate: {_SAMPLE_RATE:.1f} Hz\n"
        f"- fft_size: {analyzer.fft_size}\n"
        f"- hop_size(50%): {analyzer.hop_size}\n"
    )

    last_alarm_ts: Dict[int, float] = {1: 0.0, 2: 0.0, 3: 0.0, 4: 0.0}

    try:
        while True:
            pkt = rx.get_packet(timeout=_PACKET_TIMEOUT_S)
            if pkt is None:
                continue

            parsed = parse_packet(pkt)
            if parsed is None:
                continue

            if parsed.channel_group == "ch12":
                ch_a, ch_b = 1, 2
            else:
                ch_a, ch_b = 3, 4

            a = _bytes_to_int16_samples(parsed.channel_a_data)
            b = _bytes_to_int16_samples(parsed.channel_b_data)

            analyzer.append_samples(ch_a, a)
            analyzer.append_samples(ch_b, b)

            for ch in (ch_a, ch_b):
                while True:
                    result = analyzer.compute_if_ready(ch)
                    if result is None:
                        break

                    band_mag = result.magnitude[band_mask]
                    if band_mag.size == 0:
                        break

                    band_peak = float(np.max(band_mag))
                    if band_peak <= _THRESHOLD:
                        continue

                    now = time.time()
                    if now - last_alarm_ts[ch] < _MIN_ALARM_INTERVAL_S:
                        continue

                    cache_id_hex = parsed.audio_cache_id_raw.hex(" ")
                    print(
                        f"[ALARM] ch{ch} "
                        f"band_peak={band_peak:.3f} > {_THRESHOLD:.3f} "
                        f"(band={_BAND_START_HZ/1000:.0f}-{_BAND_END_HZ/1000:.0f}kHz) "
                        f"pkt_index={pkt.get('index')} cache_id={cache_id_hex}"
                    )
                    last_alarm_ts[ch] = now
    except KeyboardInterrupt:
        print("\n退出报警监测。")
    finally:
        rx.stop()


